<?php

namespace App\Repositories\Focus\supplier;

use App\Models\supplier\Supplier;
use App\Exceptions\GeneralException;
use App\Models\Company\Company;
use App\Models\customergroup\CustomerGroupEntry;
use App\Repositories\BaseRepository;
use App\Repositories\Focus\JournalEntryService;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

/**
 * Class SupplierRepository.
 */
class SupplierRepository extends BaseRepository
{
    use JournalEntryService, SupplierStatement;

    /**
     *customer_picture_path .
     *
     * @var string
     */
    protected $supplier_picture_path;
    /**
     * Storage Class Object.
     *
     * @var \Illuminate\Support\Facades\Storage
     */
    protected $storage;
    /**
     * Associated Repository Model.
     */
    const MODEL = Supplier::class;
    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->supplier_picture_path = 'img' . DIRECTORY_SEPARATOR . 'supplier' . DIRECTORY_SEPARATOR;
        $this->storage = Storage::disk('public');
    }
    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query()->where('user_type', 'supplier');

        return $q->get();
    }

    /**
     * For Creating the respective model in storage
     *
     * @param array $input
     * @return bool
     * @throws GeneralException
     */
    public function create(array $input)
    {
        // dd($input);
        foreach ($input as $key => $value) {
            if (in_array($key, ['credit_limit', 'open_balance', 'inv_amount', 'inv_amountpaid', 'frx_rate'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => NumberClean($v), $value);
                else $input[$key] = numberClean($value);
            } 
            if (in_array($key, ['open_balance_date', 'inv_date', 'inv_due_date'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => date_for_database($v), $value);
                else $input[$key] = date_for_database($value);
            } 
        }

        $inv_params = ['inv_no', 'inv_date', 'inv_due_date', 'inv_descr', 'inv_amount', 'inv_amountpaid'];
        $supplier_invoices = Arr::only($input, $inv_params); 
        $supplier_invoices = array_filter(modify_array($supplier_invoices), fn($v) => @$v['inv_amount'] && @$v['inv_date']); 
        $supplier_data = array_diff_key($input, array_flip(['invoice_id', 'supplier_groups',...$inv_params]));
        // dd($supplier_data, $supplier_invoices);  

        DB::beginTransaction();
         
        // validate duplicate company and email
        $is_company = Supplier::where('company', $input['company'])->exists();
        if ($is_company) throw ValidationException::withMessages(['Company already exists!']);
        $email_exists = Supplier::where('email', $input['email'])->whereNotNull('email')->exists();
        if ($email_exists) throw ValidationException::withMessages(['Duplicate email!']);

        // validate tax pin
        // if (@$input['taxid']) {
        //     $taxid_exists = Supplier::where('taxid', $input['taxid'])->whereNotNull('taxid')->exists();
        //     if ($taxid_exists) throw ValidationException::withMessages(['Duplicate Tax Pin']);
        //     $is_company = Company::where(['id' => auth()->user()->ins, 'taxid' => $input['taxid']])->whereNotNull('taxid')->exists();
        //     if ($is_company) throw ValidationException::withMessages(['Company Tax Pin is not allowed']);
        // }
        
        // picture upload
        if (@$input['picture']) $input['picture'] = $this->uploadPicture($input['picture']);

        $supplier = Supplier::create($supplier_data);
        // dd($supplier->id);
        // customer groups
        $groups = @$input['supplier_groups'] ?: [];
        $supplier_groups = array_map(fn($v) => ['customer_group_id' => $v, 'customer_id' => $supplier->id, 'ins' => $supplier->ins], array_filter($groups));
        CustomerGroupEntry::insert($supplier_groups);

        // opening balance accounting
        if ($supplier->open_balance > 0) {
            $supplier['supplier_id'] = $supplier->id;
            $supplier['inp_invoices'] = $supplier_invoices;
            $this->opening_balance_acc($supplier, 'store');
        }

        if ($supplier) {
            DB::commit();
            return $supplier;
        }
    }


    /**
     * For updating the respective Model in storage
     *
     * @param Supplier $supplier
     * @param  $input
     * @throws GeneralException
     * return bool
     */
    public function update($supplier, array $input)
    {
        // dd($input);
        foreach ($input as $key => $value) {
            if (in_array($key, ['credit_limit', 'open_balance', 'inv_amount', 'inv_amountpaid', 'frx_rate'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => NumberClean($v), $value);
                else $input[$key] = numberClean($value);
            } 
            if (in_array($key, ['open_balance_date', 'inv_date', 'inv_due_date'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => date_for_database($v), $value);
                else $input[$key] = date_for_database($value);
            } 
        }
        
        $inv_params = ['invoice_id', 'inv_no', 'inv_date', 'inv_due_date', 'inv_descr', 'inv_amount', 'inv_amountpaid'];
        $supplier_invoices = Arr::only($input, $inv_params); 
        $supplier_invoices = array_filter(modify_array($supplier_invoices), fn($v) => @$v['inv_amount'] && @$v['inv_date']); 
        $supplier_data = array_diff_key($input, array_flip(['invoice_id', 'supplier_groups', ...$inv_params]));
        // dd($supplier_data, $supplier_invoices);  

        DB::beginTransaction();

        // validate duplicate company and email
        $is_company = Supplier::where('id', '!=', $supplier->id)->where('company', $input['company'])->exists();
        if ($is_company) throw ValidationException::withMessages(['Company already exists!']);
        $email_exists = Supplier::where('id', '!=', $supplier->id)->where('email', $input['email'])->whereNotNull('email')->exists();
        if ($email_exists) throw ValidationException::withMessages(['Email already in use!']);

        // validate tax pin
        if (@$input['taxid']) {
            $taxid_exists = Supplier::where('id', '!=', $supplier->id)->where('taxid', $input['taxid'])->whereNotNull('taxid')->exists();
            if ($taxid_exists) throw ValidationException::withMessages(['Duplicate Tax Pin']);
            $is_company = Company::where(['id' => auth()->user()->ins, 'taxid' => $input['taxid']])->whereNotNull('taxid')->exists();
            if ($is_company) throw ValidationException::withMessages(['Company Tax Pin is not allowed']);
        }
        
        $result = $supplier->update($supplier_data);

        // customer groups
        $groups = @$input['supplier_groups'] ?: [];
        $supplier_groups = array_map(fn($v) => ['customer_group_id' => $v, 'customer_id' => $supplier->id, 'ins' => $supplier->ins], array_filter($groups));
        CustomerGroupEntry::whereNotIn('customer_group_id', $groups)->where('customer_id', $supplier->id)->delete();
        foreach ($supplier_groups as $group) {
            CustomerGroupEntry::updateOrCreate($group, $group);
        }

        // opening balance accounting
        $supplier['supplier_id'] = $supplier->id;
        $supplier['inp_invoices'] = $supplier_invoices;
        $this->opening_balance_acc($supplier, 'update');
        
        if ($result) {
            DB::commit();
            return $result;
        }
    }

    /**
     * For deleting the respective model from storage
     *
     * @param Supplier $supplier
     * @return bool
     * @throws GeneralException
     */
    public function delete($supplier)
    {
        DB::beginTransaction();

        // remove opening balance invoices
        $supplier['supplier_id'] = $supplier->id;
        $this->opening_balance_acc($supplier, 'delete');

        CustomerGroupEntry::where('customer_id', $supplier->id)->delete();
        if ($supplier->delete()) {
            DB::commit();
            return true;
        }
    }

    /*
    * Upload logo image
    */
    public function uploadPicture($logo)
    {
        $image_name = $this->supplier_picture_path . time() . $logo->getClientOriginalName();
        $this->storage->put($image_name, file_get_contents($logo->getRealPath()));
        return $image_name;
    }

    /*
    * remove logo or favicon icon
    */
    public function removePicture(Supplier $supplier, $type)
    {
        if ($supplier->$type) {
            $image = $this->supplier_picture_path . $supplier->type;
            if ($this->storage->exists($image)) $this->storage->delete($image);
        }
        return $supplier->update([$type => null]);
    }
}
