<?php

namespace App\Repositories\Focus\quote;

use App\Models\items\QuoteItem;
use App\Models\items\VerifiedItem;
use App\Models\quote\Quote;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use App\Models\lead\Lead;
use App\Models\project\BudgetSkillset;
use App\Models\verifiedjcs\VerifiedJc;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use App\Models\bom\BoM;
use App\Models\bom\BoMItem;
use Illuminate\Support\Arr;
use App\Models\work_order\WorkOrder;
use App\Models\work_order\WorkOrderItem;

/**
 * Class QuoteRepository.
 */
class QuoteRepository extends BaseRepository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Quote::class;

    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query();
        
        // $q->when(request('page') == 'pi', fn($q) => $q->where('bank_id', '>', 0));
        // $q->when(request('page') == 'qt', fn($q) => $q->where('bank_id', 0));
        
        $q->when(request('start_date') && request('end_date'), function ($q) {
            $q->whereBetween('date', array_map(fn($v) => date_for_database($v), [request('start_date'), request('end_date')]));
        });

        // client filter
        $q->when(request('client_id'), fn($q) => $q->where('customer_id', request('client_id')));
        
        // status criteria filter
        $status = true;
        if (request('status_filter')) {
            switch (request('status_filter')) {
                case 'Unapproved':
                    $q->whereNull('approved_by');
                    break;
                case 'Approved':
                    $q->whereNotNull('approved_by');
                    break;
                case 'Budgeted & Unverified':
                    $q->whereNotNull('project_quote_id')->whereNull('verified_by');
                    break;
                case 'Verified with LPO & Uninvoiced':
                    $q->whereNotNull('verified_by')->whereNotNull('lpo_id')->where('invoiced', 'No');
                    break;
                case 'Verified without LPO & Uninvoiced':
                    $q->whereNotNull('verified_by')->whereNull('lpo_id')->where('invoiced', 'No');
                    break;
                case 'Approved without LPO & Uninvoiced':
                    $q->whereNotNull('approved_by')->whereNull('lpo_id')->where('invoiced', 'No');
                    break;
                case 'Invoiced & Due':
                    // quotes in due invoices
                    $q->whereHas('invoice_product', function ($q) {
                        $q->whereHas('invoice', function ($q) {
                            $q->where('status', 'due');
                        });
                    });
                    break;
                case 'Invoiced & Partially Paid':
                    // quotes in partially paid invoices
                    $q->whereHas('invoice_product', function ($q) {
                        $q->whereHas('invoice', function ($q) {
                            $q->where('status', 'partial');
                        });
                    });
                    break;
                case 'Invoiced & Paid':
                    // quotes in partially paid invoices
                    $q->whereHas('invoice_product', function ($q) {
                        $q->whereHas('invoice', function ($q) {
                            $q->where('status', 'paid');
                        });
                    });
                    break;
                case 'Cancelled':
                    $status = false;
                    $q->where('status', 'cancelled');
                    break;
            }
        }
        $q->when($status, fn($q) => $q->where('status', '!=', 'cancelled'));

        // project quote filter
        $q->when(request('project_id'), function($q) {
            if (request('quote_ids')) $q->whereIn('id', explode(',', request('quote_ids')));
            else $q->whereIn('id', [0]);
        });
        
        return $q;
    }

    /**
     * Quotes pending verification
     */
    public function getForVerifyDataTable()
    {
        $q = $this->query()->where('status', '!=', 'cancelled')->where(function($q) { 
            $q->whereHas('budget')->orWhere('quote_type', 'standard'); 
        });

        $q->when(request('start_date') && request('end_date'), function ($q) {
            $q->whereBetween('date', array_map(fn($v) => date_for_database($v), [request('start_date'), request('end_date')]));
        });
        $q->when(request('customer_id'), fn($q) => $q->where('customer_id', request('customer_id')));
        $q->when(request('verify_state'), fn($q) => $q->where('verified', request('verify_state')));
        
        return $q;
    }
    public function getSelfDataTable($req)
    {
        //  dd($req);
        $q = $this->query()->withoutGlobalScopes()->where('customer_id',$req);
        //  dd($q->get());
        
        return $q->get();
    }

    /**
     * Quotes pending invoicing
     */
    public function getForVerifyNotInvoicedDataTable()
    {
        $q = $this->query();
        // verified and uninvoiced quotes
        $q->where(['verified' => 'Yes', 'invoiced' => 'No'])->whereDoesntHave('invoice_product');
                
        $q->when(request('customer_id'), fn($q) => $q->where('customer_id', request('customer_id')));
        $q->when(request('lpo_number'), fn($q) => $q->where('lpo_id', request('lpo_number')));
        $q->when(request('project_id'), function ($q) {
            $q->whereIn('id', function($q) {
                $q->select('quote_id')->from('project_quotes')->where('project_id', request('project_id'));
            });
        });
        
        return $q->get([
            'id', 'notes', 'tid', 'customer_id', 'lead_id', 'branch_id', 'date', 
            'total', 'bank_id', 'verified_total', 'lpo_id', 'project_quote_id', 'currency_id'
        ]);
    }

    /**
     * For Creating the respective model in storage
     *
     * @param array $input
     * @throws GeneralException
     * @return $quote
     */
    public function create(array $input)
    {
        // dd($input);
        DB::beginTransaction();

        // $bom = $input['bom'];
        // foreach ($bom as $key => $value) {
        //     if(in_array($key, ['bom_date']))
        //         $bom[$key] = date_for_database($value);
        //     if(in_array($key, ['total_bom']))
        //         $bom[$key] = numberClean($value);
        // }
        // $result_bom = BoM::create($bom);

        $data = $input['data'];
        foreach ($data as $key => $val) {
            if (in_array($key, ['date']))
                $data[$key] = date_for_database($val);
            if (in_array($key, ['total', 'subtotal', 'tax', 'taxable']))
                $data[$key] = numberClean($val);
        }
        // $data['bom_id'] = 0;
        // if($result_bom){
        //     $data['bom_id'] = $result_bom->id;
        // }

        $result = Quote::create($data);

        // // bom line items
        // $bom_items = $input['bom_items'];
        // $bom_items = array_map(function ($v) use($result) {
        //     return array_replace($v, [
        //         'bom_id' => $result->id, 
        //         'ins' => $result->ins,
        //         'purchase_price' =>  floatval(str_replace(',', '', $v['purchase_price'])),
        //     ]);
        // }, $bom_items);
        // BoMItem::insert($bom_items);
        // quote line items
        $data_items = $input['data_items'];
        $data_items = array_map(function ($v) use($result) {
            return array_replace($v, [
                'quote_id' => $result->id, 
                'ins' => $result->ins,
                'product_price' =>  floatval(str_replace(',', '', $v['product_price'])),
                'product_subtotal' => floatval(str_replace(',', '', $v['product_subtotal'])),
                'buy_price' => floatval(str_replace(',', '', $v['buy_price'])),
            ]);
        }, $data_items);
        QuoteItem::insert($data_items);

        
        if ($result) {
            DB::commit();
            return $result;
        }
        
        throw new GeneralException('Error Creating Quote');
    }

    /**
     * For updating the respective Model in storage
     *
     * @param Quote $quote
     * @param  $input
     * @throws GeneralException
     * @return $quote
     */
    public function update($quote, array $input)
    {
        // dd($input);
        DB::beginTransaction();

        // $bom = $input['bom'];
        // foreach ($bom as $key => $val) {
        //     if (in_array($key, ['bom_date']))
        //         $bom[$key] = date_for_database($val);
        //     if (in_array($key, ['total_bom'])) 
        //         $bom[$key] = floatval(str_replace(',', '', $val));
        // }   
        
        // $result_bom = $quote->bom->update($bom);

        $data = $input['data'];
        foreach ($data as $key => $val) {
            if (in_array($key, ['date']))
                $data[$key] = date_for_database($val);
            if (in_array($key, ['total', 'subtotal', 'tax', 'taxable'])) 
                $data[$key] = floatval(str_replace(',', '', $val));
        }   
        
        $result = $quote->update($data);

        $data_items = $input['data_items'];
        // remove omitted items
        $item_ids = array_map(function ($v) { return $v['id']; }, $data_items);
        $quote->products()->whereNotIn('id', $item_ids)->delete();

        // create or update items
        foreach($data_items as $item) {
            foreach ($item as $key => $val) {
                if (in_array($key, ['product_price', 'product_subtotal', 'buy_price']))
                    $item[$key] = floatval(str_replace(',', '', $val));
            }
            $quote_item = QuoteItem::firstOrNew(['id' => $item['id']]);
            $quote_item->fill(array_replace($item, ['quote_id' => $quote['id'], 'ins' => $quote['ins']]));
            if (!$quote_item->id) unset($quote_item->id);
            $quote_item->save();
        }

        // $bom_items = $input['bom_items'];
        // // remove omitted items
        // $bom_id = array_map(function ($v) { return $v['bom_id']; }, $bom_items);
        // $quote->bom->bom_items()->whereNotIn('id', $bom_id)->delete();
        // // create or update items
        // foreach($bom_items as $item) {
        //     $bom_item = BoMItem::firstOrNew(['id' => $item['bom_id']]);         
        //     $bom_item->fill(array_replace($item, ['bom_id' => $quote->id]));
        //     if (!$bom_item->id) unset($bom_item->id);
        //     unset($bom_item->bom_id);
        //     $bom_item->save();
        // }

        if ($result) {
            DB::commit();
            return $quote;      
        }
               
        throw new GeneralException('Error Updating Quote');
    }

    /**
     * For deleting the respective model from storage
     *
     * @param Quote $quote
     * @throws GeneralException
     * @return bool
     */
    public function delete($quote)
    {
        DB::beginTransaction();

        $type = $quote->bank_id ? 'PI' : 'Quote';
        if ($quote->project_quote) 
            throw ValidationException::withMessages([$type . ' is attached to a project!']);
            
        if ($quote->delete()) {
            if ($quote->lead) $quote->lead->update(['status' => 0, 'reason' => 'new']);
            DB::commit();
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.quotes.delete_error'));
    }

    public function create_workorder(array $input)
    {
        $input['frx_rate'] = $input['currency_rate'];
        $input['item_tax_id'] = $input['tax_rate'];
        $input['tax_rate'] = $input['tax_id'];
        $input['note'] = $input['notes'];
        $input['product_amount'] = $input['product_price'];
        $input['product_price'] = $input['buy_price'];
        $input['description'] = $input['product_name'];
        unset($input['currency_rate'],$input['tax_id'],$input['notes'],$input['product_name'],$input['buy_price']);
        // dd($input);

        DB::beginTransaction();

        foreach ($input as $key => $val) {
            if ($key == 'date') $input[$key] = date_for_database($val);
            $keys = ['total', 'subtotal', 'taxable', 'tax', 'tax_rate', 'frx_rate', 'product_qty', 'product_price', 'item_tax_id', 'product_amount'];
            if (in_array($key, $keys)) {
                if (is_array($val)) $input[$key] = array_map(fn($v) => numberClean($v), $val);
                else $input[$key] = numberClean($val);
            }
        }

        $data = Arr::only($input, [
            'customer_id', 'date', 'tax_rate', 'bank_id', 'validity', 'ledger_id', 'currency_id', 
            'frx_rate', 'term_id', 'note', 'taxable', 'subtotal', 'tax', 'total'
        ]);
        if ($data['frx_rate'] > 1) {
            $data = array_replace($data, [
                'frx_taxable' => round($data['taxable'] * $data['frx_rate'], 4),
                'frx_subtotal' => round($data['subtotal'] * $data['frx_rate'], 4),
                'frx_tax' => round($data['tax'] * $data['frx_rate'], 4),
                'frx_total' => round($data['total'] * $data['frx_rate'], 4),
            ]);
        }
        $data['due_date'] = date_for_database("{$data['date']} + {$data['validity']} days");
        $work_order = WorkOrder::create($data);

        $data_items = Arr::only($input, [
            'numbering', 'description', 'unit', 'product_qty', 'product_price', 'item_tax_id', 
            'product_amount', 'product_id', 'item_tax_id'
        ]);
        $data_items = modify_array($data_items);
        $data_items = array_filter($data_items, fn($v) => $v['product_qty']);
        if (!$data_items) throw ValidationException::withMessages(['Cannot Invoice without product line items!']);
        $data_items = array_map(function($v) use($work_order) {
            return [
                'work_order_id' => $work_order->id,
                'product_id' => $v['product_id'],
                'num' => $v['numbering'],
                'name' => $v['description'],
                'qty' => $v['product_qty'],
                'subtotal' => $v['product_price'],
                'tax' => $v['product_price'],
                'total' => $v['product_amount'] * $v['item_tax_id'],
                'tax_rate' => $v['item_tax_id'],
                'uom' => $v['unit'],
            ];
        }, $data_items);
        WorkOrderItem::insert($data_items);

        if ($work_order) {
            DB::commit();
            return $work_order;
        }
    }


    /**
     * Verify Budgeted Project Quote
     */
    public function verify(array $input)
    {
        // dd($input);
        DB::beginTransaction();

        // update quote verification status
        $data = $input['data'];
        $quote = Quote::find($data['id']);
        $result = $quote->update([
            'verified' => 'Yes', 
            'verification_date' => date('Y-m-d'),
            'verified_by' => auth()->user()->id,
            'gen_remark' => $data['gen_remark'],
            'verified_amount' => numberClean($data['subtotal']),
            'verified_total' => numberClean($data['total']),
            'verified_tax' => numberClean($data['tax']), 
            'verified_taxable' => numberClean($data['taxable']), 
        ]);

        // quote verified products
        $data_items = $input['data_items'];
        // delete omitted items
        $item_ids = array_map(fn($v) => $v['item_id'], $data_items);
        VerifiedItem::where('quote_id', $data['id'])->whereNotIn('id', $item_ids)->delete();
        // update or create verified item
        foreach ($data_items as $item) {
            $item = array_replace($item, [
                'quote_id' => $data['id'],
                'product_qty' => numberClean($item['product_qty']),
                'product_price' => floatval(str_replace(',', '', $item['product_price'])),
                'product_subtotal' => floatval(str_replace(',', '', $item['product_subtotal'])),
                'ins' => auth()->user()->ins,
            ]);
            $item['product_tax'] = $item['product_subtotal'] * $item['tax_rate'];

            $verify_item = VerifiedItem::firstOrNew(['id' => $item['item_id']]);
            $verify_item->fill($item);
            unset($verify_item->item_id);
            $verify_item->save();
        }

        // quote verified jobcards
        $job_cards = $input['job_cards'];
        // delete omitted items
        $item_ids = array_map(fn($v) => $v['jcitem_id'], $job_cards);
        VerifiedJc::where('quote_id', $data['id'])->whereNotIn('id', $item_ids)->delete();        
        // duplicate jobcard reference
        $references = array_map(fn($v) => $v['reference'], $job_cards);
        $references = VerifiedJc::whereIn('reference', $references)->pluck('reference')->toArray();
        // update or create verified jobcards
        foreach ($job_cards as $item) {
            // skip duplicate reference
            if (in_array($item['reference'], $references) && !$item['jcitem_id']) continue;
            $item = array_replace($item, [
                'quote_id' => $data['id'],
                'date' => date_for_database($item['date']),
            ]);

            $jobcard = VerifiedJc::firstOrNew(['id' => $item['jcitem_id']]);
            $jobcard->fill($item);
            unset($jobcard->jcitem_id);
            $jobcard->save();
        }

        if ($result) {
            DB::commit();
            return $quote;      
        }
        
        throw new GeneralException('Error Verifying Quote');
    }
}