<?php

namespace App\Repositories\Focus\invoice;

use App\Models\account\Account;
use App\Models\invoice\Invoice;
use App\Models\items\InvoiceItem;
use App\Models\transaction\Transaction;
use App\Repositories\BaseRepository;
use App\Models\quote\Quote;
use App\Models\transactioncategory\Transactioncategory;
use App\Models\work_order\WorkOrder;
use App\Repositories\Focus\JournalEntryService;
use DB;

/**
 * Class InvoiceRepository.
 */
class InvoiceRepository extends BaseRepository
{
    use JournalEntryService;

    /**
     * Associated Repository Model
     * 
     */
    const MODEL = Invoice::class;

    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * 
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query();

        // date filter
        if (request('start_date') && request('end_date')) {
            $q->whereBetween('date', [
                date_for_database(request('start_date')), 
                date_for_database(request('end_date'))
            ]);
        }

        // project filter
        $q->when(request('project_id'), function($q) {
            $q->whereHas('quotes', function($q) {
                $q->whereHas('project', function($q) {
                    $q->where('projects.id', request('project_id'));
                });
            });
        });


        // customer and status filter
        $q->when(request('customer_id'), function ($q) {
            $q->where('customer_id', request('customer_id'));
        })->when(request('invoice_status'), function ($q) {
            $status = request('invoice_status');
            switch ($status) {
                case 'not yet due': 
                    $q->where('due_date', '>', date('Y-m-d'));
                    break;
                case 'due':    
                    $q->where('due_date', '<=', date('Y-m-d'));
                    break;                 
            }         
        })->when(request('payment_status'), function ($q) {
            $status = request('payment_status');
            // switch ($status) {
            //     case 'unpaid':
            //         $q->where('amountpaid', 0);
            //         break; 
            //     case 'partially paid':
            //         $q->whereColumn('amountpaid', '<', 'total')->where('amountpaid', '>', 0);
            //         break; 
            //     case 'paid':
            //         $q->whereColumn('amountpaid', '>=', 'total');
            //         break; 
            // }         
        });

        return $q;
    }

    /**
     * Create Invoice
     */
    public function create(array $input)
    {
        DB::beginTransaction();

        $work_order = WorkOrder::findOrFail($input['work_order_id']);
        $work_order_data = array_diff_key($work_order->toArray(), array_flip([
            'id', 'tid', 'created_at', 'updated_at', 'status', 'status_note', 'status_mod_by', 'user_id', 'ins'
        ]));
        
        // invoice
        $inv_data = array_replace($work_order_data, [
            'doc_type' => 'invoice',
            'tid' => Invoice::max('tid')+1,
            'work_order_id' => $work_order->id,
        ]);
        $invoice = Invoice::create($inv_data);

        // invoice items
        $inv_items_data = array_map(function($v) use($invoice) {
            $v = array_diff_key($v, array_flip([
                'id', 'work_order_id', 'created_at', 'updated_at'
            ]));
            return array_replace($v, [
                'invoice_id' => $invoice->id,
            ]);
        }, $work_order->items->toArray());
        InvoiceItem::insert($inv_items_data);

        // generate invoice accounting
        $this->generate_invoice_acc($invoice, 'store');

        if ($invoice) {
            DB::commit();
            return $invoice;
        }
    }


    /**
     * Project Invoice transaction
     * 
     */
    public function post_transaction_project_invoice($result)
    {
        // debit Accounts Receivable (Debtors)
        $account = Account::where('system', 'receivable')->first(['id']);
        $tr_category = Transactioncategory::where('code', 'inv')->first(['id', 'code']);
        $tid = Transaction::where('ins', auth()->user()->ins)->max('tid') + 1;
        $dr_data = [
            'tid' => $tid,
            'account_id' => $account->id,
            'trans_category_id' => $tr_category->id,
            'debit' => $result->total,
            'tr_date' => $result->invoicedate,
            'due_date' => $result->invoiceduedate,
            'user_id' => $result->user_id,
            'note' => $result->notes,
            'ins' => $result->ins,
            'tr_type' => $tr_category->code,
            'tr_ref' => $result->id,
            'user_type' => 'customer',
            'is_primary' => 1,
        ];
        Transaction::create($dr_data);

        unset($dr_data['debit'], $dr_data['is_primary']);

        // credit Revenue Account (Income)
        $inc_cr_data = array_replace($dr_data, [
            'account_id' => $result->account_id,
            'credit' => $result->subtotal,
        ]);
        Transaction::create($inc_cr_data);

        // credit tax (VAT)
        if ($result->tax > 0) {
            $account = Account::where('system', 'tax')->first(['id']);
            $tax_cr_data = array_replace($dr_data, [
                'account_id' => $account->id,
                'credit' => $result->tax,
            ]);
            Transaction::create($tax_cr_data);
        }

        // WIP and COG transactions
        $tr_data = array();

        // stock amount for items issued from inventory
        $store_inventory_amount = 0;
        // direct purchase item amounts for item directly issued to project
        $dirpurch_inventory_amount = 0;
        $dirpurch_expense_amount = 0;
        $dirpurch_asset_amount = 0;

        // invoice related quotes and pi
        $quote_ids = $result->products->pluck('quote_id')->toArray();
        $quotes = Quote::whereIn('id', $quote_ids)->get();
        foreach ($quotes as $quote) {
            $store_inventory_amount  = $quote->projectstock->sum('subtotal');
            // direct purchase items issued to project
            if (isset($quote->project_quote->project)) {
                foreach ($quote->project_quote->project->purchase_items as $item) {
                    if ($item->itemproject_id) {
                        $subtotal = $item->amount - $item->taxrate;
                        if ($item->type == 'Expense') $dirpurch_expense_amount += $subtotal;
                        elseif ($item->type == 'Stock') $dirpurch_inventory_amount += $subtotal;
                        elseif ($item->type == 'Asset') $dirpurch_asset_amount += $subtotal;
                    }
                    
                }
            }
        }

        // credit WIP account and debit COG
        $wip_account = Account::where('system', 'wip')->first(['id']);
        $cog_account = Account::where('system', 'cog')->first(['id']);
        $cr_data = array_replace($dr_data, ['account_id' => $wip_account->id, 'is_primary' => 1]);
        $dr_data = array_replace($dr_data, ['account_id' => $cog_account->id, 'is_primary' => 0]);
        
        if ($dirpurch_inventory_amount > 0) {
            $tr_data[] = array_replace($cr_data, ['credit' => $dirpurch_inventory_amount]);
            $tr_data[] = array_replace($dr_data, ['debit' => $dirpurch_inventory_amount]);
        }
        if ($dirpurch_expense_amount > 0) {
            $tr_data[] = array_replace($cr_data, ['credit' => $dirpurch_expense_amount]);
            $tr_data[] = array_replace($dr_data, ['debit' => $dirpurch_expense_amount]);
        }
        if ($dirpurch_asset_amount > 0) {
            $tr_data[] = array_replace($cr_data, ['credit' => $dirpurch_asset_amount]);
            $tr_data[] = array_replace($dr_data, ['debit' => $dirpurch_asset_amount]);
        }
        if ($store_inventory_amount > 0) {
            $tr_data[] = array_replace($cr_data, ['credit' => $store_inventory_amount]);
            $tr_data[] = array_replace($dr_data, ['debit' => $store_inventory_amount]);
        }

        $tr_data = array_map(function ($v) {
            if (isset($v['debit']) && $v['debit'] > 0) $v['credit'] = 0;
            elseif (isset($v['credit']) && $v['credit'] > 0) $v['debit'] = 0;
            return $v;
        }, $tr_data);

        Transaction::insert($tr_data);        
        aggregate_account_transactions();        
    }
}

