<?php

namespace App\Repositories\Focus\customer;

use App\Models\customer\Customer;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Storage;
use App\Models\branch\Branch;
use App\Models\Company\Company;
use App\Models\customergroup\CustomerGroupEntry;
use App\Models\invoice\Invoice;
use App\Repositories\Focus\JournalEntryService;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Hash;

/**
 * Class CustomerRepository.
 */
class CustomerRepository extends BaseRepository
{
    use JournalEntryService, CustomerStatement;

    /**
     *customer_picture_path .
     *
     * @var string
     */
    protected $customer_picture_path;


    /**
     * Storage Class Object.
     *
     * @var \Illuminate\Support\Facades\Storage
     */
    protected $storage;

    /**
     * Associated Repository Model.
     */
    const MODEL = Customer::class;


    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->customer_picture_path = 'img' . DIRECTORY_SEPARATOR . 'customer' . DIRECTORY_SEPARATOR;
        $this->storage = Storage::disk('public');
    }

    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query()->where('user_type', 'customer');

        return $q->get(['id', 'name', 'company', 'email', 'address', 'picture', 'created_at']);
    }

    /**
     * Customer Invoices data
     */
    public function getInvoicesForDataTable($customer_id = 0)
    {
        return Invoice::where('customer_id', request('customer_id', $customer_id))->get();
    }

    /**
     * For Creating the respective model in storage
     *
     * @param array $input
     * @return bool
     * @throws GeneralException
     */
    public function create(array $input)
    {
        // dd($input);  
        foreach ($input as $key => $value) {
            if (in_array($key, ['credit_limit', 'open_balance', 'inv_amount', 'inv_amountpaid', 'frx_rate'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => NumberClean($v), $value);
                else $input[$key] = numberClean($value);
            } 
            if (in_array($key, ['open_balance_date', 'inv_date', 'inv_due_date'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => date_for_database($v), $value);
                else $input[$key] = date_for_database($value);
            } 
        }
        
        $inv_params = ['inv_no', 'inv_date', 'inv_due_date', 'inv_descr', 'inv_amount', 'inv_amountpaid'];
        $branch_params = ['branch_code', 'branch_name', 'branch_location', 'branch_cp', 'branch_cp_no'];
        $customer_invoices = Arr::only($input, $inv_params); 
        $customer_invoices = array_filter(modify_array($customer_invoices), fn($v) => @$v['inv_amount'] && @$v['inv_date']); 
        $customer_branches = Arr::only($input, $branch_params); 
        $customer_branches = array_filter(modify_array($customer_branches), fn($v) => @$v['branch_name'] && @$v['branch_location']); 
        $customer_data = array_diff_key($input, array_flip(['invoice_id', 'branch_id', 'customer_groups',...$inv_params, ...$branch_params]));
        // dd($customer_data, $customer_invoices, $customer_branches);  

        DB::beginTransaction();
         
        // validate duplicate company and email
        $is_company = Customer::where('company', $input['company'])->exists();
        if ($is_company) throw ValidationException::withMessages(['Company already exists!']);
        $email_exists = Customer::where('email', $input['email'])->whereNotNull('email')->exists();
        if ($email_exists) throw ValidationException::withMessages(['Duplicate email!']);

        // validate tax pin
        // if (@$input['taxid']) {
        //     $taxid_exists = Customer::where('taxid', $input['taxid'])->whereNotNull('taxid')->exists();
        //     if ($taxid_exists) throw ValidationException::withMessages(['Duplicate Tax Pin']);
        //     $is_company = Company::where(['id' => auth()->user()->ins, 'taxid' => $input['taxid']])->whereNotNull('taxid')->exists();
        //     if ($is_company) throw ValidationException::withMessages(['Company Tax Pin is not allowed']);
        // }
        
        // picture upload
        if (@$input['picture']) $input['picture'] = $this->uploadPicture($input['picture']);

        $customer = Customer::create($customer_data);

        // customer groups
        $groups = @$input['customer_groups'] ?: [];
        $customer_groups = array_map(fn($v) => ['customer_group_id' => $v, 'customer_id' => $customer->id, 'ins' => $customer->ins], array_filter($groups));
        CustomerGroupEntry::insert($customer_groups);

        // branches
        $customer_branches = array_map(fn($v) => [
            'branch_code' => $v['branch_code'],
            'name' => $v['branch_name'],
            'location' => $v['branch_location'],
            'contact_name' => $v['branch_cp'],
            'contact_phone' => $v['branch_cp_no'],
            'customer_id' => $customer->id, 
            'user_id' => $customer->user_id,
            'ins' => $customer->ins,
        ], $customer_branches);
        Branch::insert($customer_branches);

        // opening balance accounting
        if ($customer->open_balance > 0) {
            $customer['customer_id'] = $customer->id;
            $customer['inp_invoices'] = $customer_invoices;
            $this->opening_balance_acc($customer, 'store');
        }

        if ($customer) {
            DB::commit();
            return $customer;
        }
    }

    /**
     * For updating the respective Model in storage
     *
     * @param Customer $customer
     * @param  $input
     * @throws GeneralException
     * return bool
     */
    public function update($customer, array $input)
    {
        // dd($input);
        foreach ($input as $key => $value) {
            if (in_array($key, ['credit_limit', 'open_balance', 'inv_amount', 'inv_amountpaid', 'frx_rate'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => NumberClean($v), $value);
                else $input[$key] = numberClean($value);
            } 
            if (in_array($key, ['open_balance_date', 'inv_date', 'inv_due_date'])) {
                if (is_array($value)) $input[$key] = array_map(fn($v) => date_for_database($v), $value);
                else $input[$key] = date_for_database($value);
            } 
        }
        
        $inv_params = ['invoice_id', 'inv_no', 'inv_date', 'inv_due_date', 'inv_descr', 'inv_amount', 'inv_amountpaid'];
        $branch_params = ['branch_id', 'branch_code', 'branch_name', 'branch_location', 'branch_cp', 'branch_cp_no'];
        $customer_invoices = Arr::only($input, $inv_params); 
        $customer_invoices = array_filter(modify_array($customer_invoices), fn($v) => @$v['inv_amount'] && @$v['inv_date']); 
        $customer_branches = Arr::only($input, $branch_params); 
        $customer_branches = array_filter(modify_array($customer_branches), fn($v) => @$v['branch_name'] && @$v['branch_location']); 
        $customer_data = array_diff_key($input, array_flip(['invoice_id', 'branch_id', 'customer_groups', ...$inv_params, ...$branch_params]));
        // dd($customer_data, $customer_invoices, $customer_branches);  

        DB::beginTransaction();

        // validate duplicate company and email
        $is_company = Customer::where('id', '!=', $customer->id)->where('company', $input['company'])->exists();
        if ($is_company) throw ValidationException::withMessages(['Company already exists!']);
        $email_exists = Customer::where('id', '!=', $customer->id)->where('email', $input['email'])->whereNotNull('email')->exists();
        if ($email_exists) throw ValidationException::withMessages(['Email already in use!']);

        // validate tax pin
        if (@$input['taxid']) {
            $taxid_exists = Customer::where('id', '!=', $customer->id)->where('taxid', $input['taxid'])->whereNotNull('taxid')->exists();
            if ($taxid_exists) throw ValidationException::withMessages(['Duplicate Tax Pin']);
            $is_company = Company::where(['id' => auth()->user()->ins, 'taxid' => $input['taxid']])->whereNotNull('taxid')->exists();
            if ($is_company) throw ValidationException::withMessages(['Company Tax Pin is not allowed']);
        }
        
        $result = $customer->update($customer_data);

        // customer groups
        $groups = @$input['customer_groups'] ?: [];
        $customer_groups = array_map(fn($v) => ['customer_group_id' => $v, 'customer_id' => $customer->id, 'ins' => $customer->ins], array_filter($groups));
        CustomerGroupEntry::whereNotIn('customer_group_id', $groups)->where('customer_id', $customer->id)->delete();
        foreach ($customer_groups as $group) {
            CustomerGroupEntry::updateOrCreate($group, $group);
        }

        // update branches
        $customer_branches = array_map(fn($v) => [
            'id' => $v['branch_id'],
            'branch_code' => $v['branch_code'],
            'name' => $v['branch_name'],
            'location' => $v['branch_location'],
            'contact_name' => $v['branch_cp'],
            'contact_phone' => $v['branch_cp_no'],
            'customer_id' => $customer->id, 
            'user_id' => $customer->user_id,
            'ins' => $customer->ins,
        ], $customer_branches);
        foreach ($customer_branches as $branch_data) {
            $branch = Branch::firstOrNew(['id' => $branch_data['id']]);
            $branch->fill($branch_data);
            $branch->save();
        }

        // opening balance accounting
        $customer['customer_id'] = $customer->id;
        $customer['inp_invoices'] = $customer_invoices;
        $this->opening_balance_acc($customer, 'update');

        if ($result) {
            DB::commit();
            return $result;
        }
    }

    /**
     * For deleting the respective model from storage
     *
     * @param Customer $customer
     * @return bool
     * @throws GeneralException
     */
    public function delete($customer)
    {
        DB::beginTransaction();

        if ($customer->leads()->exists()) 
            throw ValidationException::withMessages(['Customer has attached Tickets']);

        // remove opening balance invoices
        $customer['customer_id'] = $customer->id;
        $this->opening_balance_acc($customer, 'delete');

        CustomerGroupEntry::where('customer_id', $customer->id)->delete();
        $customer->branches()->delete();
        if ($customer->delete()) {
            DB::commit();
            return true;
        }
    }

    /*
    * Upload logo image
    */
    public function uploadPicture($file)
    {
        $image = time() . $file->getClientOriginalName();
        $this->storage->put($this->customer_picture_path . $image, file_get_contents($file->getRealPath()));
        return $image;
    }

    /*
    * Remove logo or favicon icon
    */
    public function removePicture(Customer $customer, $type)
    {
        $path = $this->customer_picture_path;
        $storage_exists = $this->storage->exists($path . $customer->$type);
        if ($customer->$type && $storage_exists)
            $this->storage->delete($path . $customer->$type);
            
        return $customer->update([$type => null]); 
    }
}
