<?php

namespace App\Repositories\Focus\account;

use App\Models\account\Account;
use App\Exceptions\GeneralException;
use App\Models\project\Project;
use App\Repositories\BaseRepository;
use App\Repositories\Focus\JournalEntryService;
use Illuminate\Validation\ValidationException;

/**
 * Class AccountRepository.
 */
class AccountRepository extends BaseRepository
{
  use JournalEntryService;
  /**
   * Associated Repository Model.
   */
  const MODEL = Account::class;
  /**
   * This method is used by Table Controller
   * For getting the table data to show in
   * the grid
   * @return mixed
   */
  public function getForDataTable()
  {
    return $this->query()->get();
  }

  /**
   * Project Gross Profit data set
   */
  public function getForProjectGrossProfit()
  {
    $q = Project::query();

    $q->when(request('start_date') && request('end_date'), function ($q) {
      $q->whereBetween('start_date', array_map(fn ($v) => date_for_database($v), [request('start_date'), request('end_date')]));
    });

    $q->when(request('status') == 'active', function ($q) {
      $q->whereHas('quotes', function ($q) {
        $q->whereHas('budget')->where('verified', 'No');
      });
    })->when(request('status') == 'complete', function ($q) {
      $q->whereHas('quotes', function ($q) {
        $q->whereHas('budget')->where('verified', 'Yes');
      });
    });

    $q->with(['customer_project', 'quotes', 'purchase_items']);

    return $q->get();
  }


  /**
   * For Creating the respective model in storage
   *
   * @param array $input
   * @throws GeneralException
   * @return bool
   */
  public function create(array $input)
  {
    // dd($input);
    $reserved_words = ['receivable', 'payable', 'inventory'];
    foreach ($reserved_words as $word) {
      if (stripos($input['name'], $word) !== false && empty($input['ledger_id']))
        throw ValidationException::withMessages(['Naming conflict. Use different Account Name']);
    }
    
    $account = Account::create($input);

    return $account;
  }

  /**
   * For updating the respective Model in storage
   *
   * @param Account $account
   * @param  $input
   * @throws GeneralException
   * @return bool
   */
  public function update($account, array $input)
  {
    // dd($input);
    $reserved_words = ['receivable', 'payable', 'inventory'];
    foreach ($reserved_words as $word) {
      if (stripos($input['name'], $word) !== false && empty($input['ledger_id'])) {
        if (!$account->system) 
          throw ValidationException::withMessages(['Naming conflict. Use different Account Name']);
      }
    }

    $input['ledger_id'] = @$input['ledger_id'];
    $result = $account->update($input);
    return $result;
  }

  /**
   * For deleting the respective model from storage
   *
   * @param Account $account
   * @throws GeneralException
   * @return bool
   */
  public function delete($account)
  {
    if ($account->transactions->exist()) throw ValidationException::withMessages(['Account has attached transactions']);
    if ($account->system) throw ValidationException::withMessages(['System account cannot be deleted']);
    $result = $account->delete();
    return $result;
  }
}
