<?php
/*
 * Rose Business Suite - Accounting, CRM and POS Software
 * Copyright (c) UltimateKode.com. All Rights Reserved
 * ***********************************************************************
 *
 *  Email: support@ultimatekode.com
 *  Website: https://www.ultimatekode.com
 *
 *  ************************************************************************
 *  * This software is furnished under a license and may be used and copied
 *  * only  in  accordance  with  the  terms  of such  license and with the
 *  * inclusion of the above copyright notice.
 *  * If you Purchased from Codecanyon, Please read the full License from
 *  * here- http://codecanyon.net/licenses/standard/
 * ***********************************************************************
 */

namespace App\Http\Controllers\Focus\product;

use App\Models\product\Product;
use App\Models\product\ProductVariation;
use App\Models\productcategory\Productcategory;
use App\Models\warehouse\Warehouse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\RedirectResponse;
use App\Http\Responses\ViewResponse;
use App\Http\Responses\Focus\product\CreateResponse;
use App\Http\Responses\Focus\product\CreateModalResponse;
use App\Http\Responses\Focus\product\EditResponse;
use App\Repositories\Focus\product\ProductRepository;
use App\Http\Requests\Focus\product\ManageProductRequest;
use App\Http\Requests\Focus\product\CreateProductRequest;
use App\Http\Requests\Focus\product\EditProductRequest;
use App\Models\client_product\ClientProduct;
use App\Models\supplier_product\SupplierProduct;
use App\Models\product\ProductMeta;
use App\Models\paper\Paper;
use App\Models\paper\PaperName;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\jobcard\JobCard;

/**
 * ProductsController
 */
class ProductsController extends Controller
{
    /**
     * variable to store the repository object
     * @var ProductRepository
     */
    protected $repository;

    /**
     * contructor to initialize repository object
     * @param ProductRepository $repository ;
     */
    public function __construct(ProductRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param App\Http\Requests\Focus\product\ManageProductRequest $request
     * @return \App\Http\Responses\ViewResponse
     */
    public function index(ManageProductRequest $request)
    {
        $warehouses = Warehouse::get(['id', 'title']);
        $categories = Productcategory::get(['id', 'title']);

        return new ViewResponse('focus.products.index', compact('warehouses', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param CreateProductRequestNamespace $request
     * @return \App\Http\Responses\Focus\product\CreateResponse
     */
    public function create(CreateProductRequest $request)
    {
        return new CreateResponse('focus.products.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreProductRequestNamespace $request
     * @return \App\Http\Responses\RedirectResponse
     */
    public function store(Request $request)
    {
        //dd($request->all());
        try {
            $this->repository->create($request->except(['_token']));
        } catch (\Throwable $th) {dd($th);
            return errorHandler($th, 'Error Creating Product');
        }

        return new RedirectResponse(route('biller.products.index'), ['flash_success' => trans('alerts.backend.products.created')]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param App\Models\product\Product $product
     * @param EditProductRequestNamespace $request
     * @return \App\Http\Responses\Focus\product\EditResponse
     */
    public function edit(Product $product, EditProductRequest $request)
    {
        return new EditResponse($product);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateProductRequestNamespace $request
     * @param App\Models\product\Product $product
     * @return \App\Http\Responses\RedirectResponse
     */
    public function update(EditProductRequest $request, Product $product)
    {
        try {
            $this->repository->update($product, $request->except(['_token']));
        } catch (\Throwable $th) {
            return errorHandler($th, 'Error Updating Product');
        }
        
        return new RedirectResponse(route('biller.products.index'), ['flash_success' => trans('alerts.backend.products.updated')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Models\product\Product $product
     * @return \App\Http\Responses\RedirectResponse
     */
    public function destroy(Product $product)
    {
        try {
            $this->repository->delete($product);
        } catch (\Throwable $th) {
            return errorHandler($th, 'Error Deleting Product');
        }
        
        return json_encode(['status' => 'Success', 'message' => trans('alerts.backend.products.deleted')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteProductRequestNamespace $request
     * @param App\Models\product\Product $product
     * @return \App\Http\Responses\RedirectResponse
     */
    public function show(Product $product, ManageProductRequest $request)
    {
        return new ViewResponse('focus.products.view', compact('product'));
    }

    /**
     * Quote or PI searchable product drop down options
     */
    public function quote_product_search(Request $request)
    {
        if (!access()->allow('product_search')) return false;

       // fetch inventory products
       $products = Product::where('name', 'LIKE', '%' . request('keyword') . '%')
       ->with(['warehouse' => function ($q) {
           $q->select(['id', 'title']);
       }])->limit(6)->get()->unique('name');
       
       $product_items = array();
       foreach ($products as $row) {
           $product = array_intersect_key($row->toArray(), array_flip([
               'id', 'name', 'code', 'qty', 'image', 'purchase_price', 'sku', 'alert'
           ]));
           $product = $product + [
               'product_des' => $row->product_des,
               'uom' => $row->unit->code
           ];
           // purchase price set by inventory valuation (LIFO) method
          // $product['purchase_price'] = $this->repository->eval_purchase_price($row->id, $row->qty, $row->purchase_price);
               
           $product_items[] =  $product;
        }
        return response()->json($products);
    }
    
    public function purchase_search(Request $request)
    {
        if (!access()->allow('product_search')) return false;

        // fetch inventory products
        $products = Product::where('name', 'LIKE', '%' . request('keyword') . '%')->where('paper_id','=','0')
        ->with(['warehouse' => function ($q) {
            $q->select(['id', 'title']);
        }])->limit(6)->get()->unique('name');
        
        $product_items = array();
        foreach ($products as $row) {
            $product = array_intersect_key($row->toArray(), array_flip([
                'id', 'name', 'code', 'qty', 'image', 'purchase_price', 'sku', 'alert'
            ]));
            // dd($row->product_bin->first()->balance);
            $balance = $row->product_bin->first() ? $row->product_bin->first()->balance : 0;
            $product = $product + [
                'product_des' => $row->product_des,
                'uom' => $row->unit ? $row->unit->code : '',
                'supplier_id' => $row->supplier_id,
                'unit' => $row->unit ? $row->unit->id : 0,
                'qty_stock' =>  NumberFormat($balance),
                'warehouse' => $row->warehouse ? $row->warehouse->title : '',
            ];
            // purchase price set by inventory valuation (LIFO) method
           // $product['purchase_price'] = $this->repository->eval_purchase_price($row->id, $row->qty, $row->purchase_price);
                
            $product_items[] =  $product;
        }

        return response()->json($product_items);
    }
    public function paper_search(Request $request)
    {
        if (!access()->allow('product_search')) return false;

        // fetch inventory products
        $products = Paper::whereHas('paper_width', function($q){
            $q->where('name','LIKE', '%' . request('keyword') . '%');
        })->with(['gsm' => function ($q) {
            $q->select(['id', 'name']);
        }])->limit(6)->get()->unique('name');
        
        $product_items = array();
        foreach ($products as $row) {
           // dd($row);
            $product = array_intersect_key($row->toArray(), array_flip([
                'id', 'name'
            ]));
            $name = $row->paper_width ? $row->paper_width->name : '';
            $gsm = $row->gsm ? $row->gsm->name : '';
            $supplier = $row->supplier ? $row->supplier->name : '';
            $product = $product + [
                'name' => $name.'-'.$gsm.'-'.$supplier,
                'uom' => $row->unit? $row->unit->code: '',
                'supplier_id' => $row->supplier_id,
                'unit' => $row->unit? $row->unit->id: 0,
                'qty_stock' => $row->product ? NumberFormat($row->product->sum('qty')) : '',
                'purchase_price' => $row->price,
            ];
            //purchase price set by inventory valuation (LIFO) method
           //$product['purchase_price'] = $this->repository->eval_purchase_price($row->id, $row->qty, $row->purchase_price);
                
           $product_items[] =  $product;
        }

        return response()->json($product_items);
    }

    public function waste_search()
    {
        // dd(request('customer_id'));
         // fetch inventory products
         $products = Product::where('name', 'LIKE', '%' . request('keyword') . '%')->where('stock_type','waste')
         ->with(['warehouse' => function ($q) {
             $q->select(['id', 'title']);
         }])->limit(6)->get()->unique('name');
         
         $product_items = array();
         foreach ($products as $row) {
             $product = array_intersect_key($row->toArray(), array_flip([
                 'id', 'name', 'code', 'qty', 'image', 'purchase_price', 'sku', 'alert'
             ]));

             $product = $product + [
                 'product_des' => $row->product_des,
                 'uom' => $row->unit ? $row->unit->code : '',
                 'unit' => $row->unit ? $row->unit->id : 0,
                 'warehouse' => $row->warehouse ? $row->warehouse->title : '',
             ];
                 
             $product_items[] =  $product;
         }
 
         return response()->json($product_items);
    }
    public function jobcard_search()
    {
        // dd(request('item_type'));
         // fetch inventory products
         if(request('item_type') == 'jobcard'){

             $jobcards = JobCard::where('jobcard_no', 'LIKE', '%' . request('keyword') . '%')->where('customer_id',request('customer_id'))
             ->where('job_status', 'complete')->orwhere('job_status', 'dispatched')
             ->where('net_qty', '>', '0')
             ->get();
             $jobcards = $jobcards->map(function($v) {
                $v->remains = $v->net_qty - $v->dispatched_qty;
                $v->item_id = 0;
                $v->type = 'jobcard';
                $v->jobcard_id = $v->id;
                return $v;
            });
         }else if(request('item_type') == 'waste'){
            $jobcards = Product::where('name', 'LIKE', '%' . request('keyword') . '%')->where('stock_type','waste')
            ->get();
            $jobcards = $jobcards->map(function($v) {
                $v->remains = $v->qty;
                $v->jobcard_no = $v->name;
                $v->item_id = $v->id;
                $v->type = 'waste';
                $v->jobcard_id = 0;
                return $v;
            });
         }
         
        

        // dd($jobcards);
         return response()->json($jobcards);
    }

    // public function combine_search(Request $request){
    //     $searchTerm = $request->keyword;
    //     $results = DB::table('products')
    //                 ->select('name', 'uom','') // Replace with the actual columns you want to select
    //                 ->where('name', 'like', "%$searchTerm%")
    //                 ->orWhere('column2', 'like', "%$searchTerm%")
    //                 ->union(
    //                     DB::table('table2')
    //                         ->select('column1', 'column2') // Replace with the actual columns you want to select
    //                         ->where('column1', 'like', "%$searchTerm%")
    //                         ->orWhere('column2', 'like', "%$searchTerm%")
    //                 )
    //                 ->get();
    // }
    // 
    // public function select(ManageProductRequest $request){
    //     $q = $request->keyword;
    //     $products = Product::where('name', 'LIKE', '%'.$q.'%')
    //         ->orderBy('name', 'asc')
    //         // ->where('active', 1)->orWhere('email', 'LIKE', '%'.$q.'')
    //         ->limit(20)->get();
    //     // foreach($products as $product){
    //     //     // $pp = ProductVariation::where('parent_id', $product->id)->first();
    //     //     $product['purchase_price'] = DB::table('product_variations')->where('parent_id', $product->id)->value('purchase_price');
    //     //     $product['selling_price'] = DB::table('product_variations')->where('parent_id', $product->id)->value('recommended_selling_price')
    //     //                                     ? DB::table('product_variations')->where('parent_id', $product->id)->value('recommended_selling_price')
    //     //                                     : 0;
    //     //     // $product['purchase_price'] = $pp->purchase_price ? $pp->purchase_price : 0;
    //     // }
    //     return response()->json($products);
    // }
    public function select(Request $request){
        if (!access()->allow('product_search')) return false;

        // fetch inventory products
        $products = Product::where('name', 'LIKE', '%' . request('keyword') . '%')->where('paper_id','>','0')
        ->with(['warehouse' => function ($q) {
            $q->select(['id', 'title']);
        }])->limit(6)->get();
        
        $product_items = array();
        foreach ($products as $row) {
            $product = array_intersect_key($row->toArray(), array_flip([
                'id', 'name', 'code', 'qty', 'image', 'purchase_price', 'sku', 'alert'
            ]));
            // dd($row->product_bin->first()->balance);
            $balance = $row->product_bin->first() ? $row->product_bin->first()->balance : 0;
            $product = $product + [
                'product_des' => $row->product_des,
                'uom' => $row->unit ? $row->unit->code : '',
                'supplier_id' => $row->supplier_id,
                'unit' => $row->unit ? $row->unit->id : 0,
                'qty_stock' =>  NumberFormat($row->weight),
                'warehouse' => $row->warehouse ? $row->warehouse->title : '',
            ];
            // purchase price set by inventory valuation (LIFO) method
           // $product['purchase_price'] = $this->repository->eval_purchase_price($row->id, $row->qty, $row->purchase_price);
                
            $product_items[] =  $product;
        }

        return response()->json($product_items);
    }
    public function product_sub_load(Request $request)
    {
        $q = $request->get('id');
        $result = \App\Models\productcategory\Productcategory::all()->where('c_type', '=', 1)->where('rel_id', '=', $q);

        return json_encode($result);
    }

    // 
    public function quick_add(CreateProductRequest $request)
    {
        return new CreateModalResponse('focus.modal.product');
    }

    /**
     * Point of Sale
     */
    public function pos(Request $request, $bill_type)
    {
        if (!access()->allow('pos')) return false;
        
        $input = $request->except('_token');
        $limit = $request->post('search_limit', 20);
        $bill_type = $request->bill_type ?: $request->type;

        if ($bill_type == 'label' && isset($input['product']['term']))
            $input['keyword'] = $input['product']['term'];

        if ($input['serial_mode'] == 1 && $input['keyword']) {
            $products = ProductMeta::where('value', 'LIKE', '%' . $input['keyword'] . '%')
                ->whereNull('value2')
                ->whereHas('product_serial', function ($q) use ($input) {
                    if ($input['wid'] > 0) $q->where('warehouse_id', $input['wid']);
                })->with(['product_standard'])->limit($limit)->get();

            $output = array();
            foreach ($products as $row) {
                $serial_product = $row->product_serial;
                $stock_product = $serial_product->product;
                $output[] = [
                    'name' => $stock_product->name, 
                    'disrate' => $serial_product->disrate, 
                    'purchase_price' => $this->repository->eval_purchase_price(
                        $stock_product->id, $stock_product->qty, $stock_product->purchase_price
                    ),
                    'price' => $serial_product['price'], 
                    'id' => $serial_product['id'], 
                    'taxrate' => $stock_product['taxrate'], 
                    'product_des' => $stock_product['product_des'], 
                    'units' => $stock_product['units'], 
                    'code' => $serial_product['code'], 
                    'alert' => $serial_product['qty'], 
                    'image' => $serial_product['image'], 
                    'serial' => $row->value,
                ];
            }
        } else {
            $products = ProductVariation::whereHas('product', function ($q) use ($input) {
                $q->where('name', 'LIKE', '%' . $input['keyword'] . '%');
                if ($input['cat_id'] > 0) $q->where('productcategory_id', $input['cat_id']);
            })->when($input['wid'] > 0, function ($q) use ($input) {
                $q->where('warehouse_id', $input['wid']);
            })->limit($limit)->get();

            $output = array();
            foreach ($products as $row) {
                $output[] = [
                    'name' => $row->name ?: $row->product->name,
                    'disrate' => numberFormat($row->disrate),
                    'purchase_price' => $this->repository->eval_purchase_price(
                        $row->id, $row->qty, $row->purchase_price
                    ),
                    'price' => numberFormat($row->price), 
                    'id' => $row->id, 
                    'taxrate' => numberFormat($row->product['taxrate']), 
                    'product_des' => $row->product['product_des'], 
                    'units' => $row->product->units, 
                    'code' => $row->code, 
                    'alert' => $row->qty, 
                    'image' => $row->image, 
                    'serial' => '',
                ];
            }
        }
        
        return view('focus.products.partials.pos')->withDetails($output);
    }
    public function getProductsForBarCodes(){
        $warehouses = Warehouse::get(['id', 'title']);
        $categories = Productcategory::get(['id', 'title']);
        return view('focus.products.barcode', compact('warehouses', 'categories'));
    }

    public function getProductDetails(Request $request){
        $product = Product::where('id', $request->product)->first();
        $product['uom'] = $product->unit ? $product->unit->code : '';
        return response()->json($product);
    }

    public function print_paper_inventory(){

        // Paper Inventory
        $reel_size = PaperName::where('type','1')->get();
        $gsm = PaperName::where('type','2')->get();
        $papers = Paper::all();
        $paper_grouped = Paper::groupBy('supplier_id', 'gsm_id')->get();

        $pdfContent = Pdf::loadView('focus.bill.print_paper_inventory', [
            'reel_size' => $reel_size,
            'gsm' => $gsm,
            'papers' => $papers,
            'paper_grouped' => $paper_grouped
          ])
            ->setPaper('a4', 'landscape')
            ->set_option('enable_css', true)
            ->set_option('css_preload', true)
            ->output();

          return response()->streamDownload(
            fn() => print $pdfContent,
            'product_barcodes.pdf',
            [
              'Content-Type' => 'application/pdf',
            ],
            'inline'
          );
    }
}
