<?php

namespace App\Http\Controllers\Focus\debitnote;

use App\Http\Controllers\Controller;
use App\Http\Responses\RedirectResponse;
use App\Http\Responses\ViewResponse;
use App\Models\debitnote\DebitNote;
use App\Models\product\ProductVariation;
use App\Repositories\Focus\debitnote\DebitNoteRepository;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class DebitNotesController extends Controller
{
  /**
   * variable to store the repository object
   * @var DebitNoteRepository
   */
  protected $repository;

  /**
   * contructor to initialize repository object
   * @param DebitNoteRepository $repository ;
   */
  public function __construct(DebitNoteRepository $repository)
  {
    $this->repository = $repository;
  }

  /**
   * Display a listing of the resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function index()
  {
    $is_debit = request('is_debit');

    return new ViewResponse('focus.debitnotes.index', compact('is_debit'));
  }

  /**
   * Show the form for creating a new resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function create()
  {
    return new ViewResponse('focus.debitnotes.create');
  }

  /**
   * Store a newly created resource in storage.
   *
   * @param  \Illuminate\Http\Request  $request
   * @return \Illuminate\Http\Response
   */
  public function store(Request $request)
  {
    try {
      $this->repository->create($request->except('_token'));
      
      $msg = 'Debit Note created successfully';
      $route = route('biller.debitnotes.index');
    } catch (\Throwable $th) {
      return errorHandler('Error Creating Debit Note', $th);
    }

    return new RedirectResponse($route, ['flash_success' => $msg]);
  }

  /**
   * Display the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function show($id)
  {
    //
  }

  /**
   * Show the form for editing the specified resource.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function edit(DebitNote $debitnote)
  {
    $is_debit = $debitnote->is_debit;
    foreach ($debitnote as $key => $val) {
      if (in_array($key, ['subtotal', 'tax', 'total'], 1)) {
        $debitnote[$key] = numberFormat($val);
      }
    }

    $prefixes = prefixesArray(['credit_note', 'debit_note'], $debitnote->ins);

    return new ViewResponse('focus.debitnotes.edit', compact('debitnote', 'is_debit', 'prefixes'));
  }

  /**
   * Update the specified resource in storage.
   *
   * @param  \Illuminate\Http\Request  $request
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function update(DebitNote $debitnote, Request $request)
  {
    // extract input fields
    $data = $request->except('_token', 'tax_id', 'amount');

    $data['ins'] = auth()->user()->ins;
    $data['user_id'] = auth()->user()->id;

    try {
      $this->repository->update($debitnote, $data);

      $msg = 'Debit Note updated successfully';
      $route = route('biller.debitnotes.index');
      if ($debitnote['is_debit']) {
        $msg = 'Debit Note updated successfully';
        $route = route('biller.debitnotes.index', 'is_debit=1');
      }
    } catch (\Throwable $th) {
      return errorHandler('Error Updating Debit Note', $th);
    }

    return new RedirectResponse($route, ['flash_success' => $msg]);
  }

  /**
   * Remove the specified resource from storage.
   *
   * @param  int  $id
   * @return \Illuminate\Http\Response
   */
  public function destroy(DebitNote $debitnote)
  {
    try {
      $this->repository->delete($debitnote);

      $msg = 'Debit Note deleted successfully';
      $route = route('biller.debitnotes.index');
      // if ($debitnote['is_debit']) {
      //   $msg = 'Debit Note updated successfully';
      //   $route = route('biller.debitnotes.index', 'is_debit=1');
      // }
    } catch (\Throwable $th) { dd($th);
      return errorHandler('Error Deleting Debit Note', $th);
    }

    return new RedirectResponse($route, ['flash_success' => $msg]);
  }

  /**
   * Print Debit Note
   */
  public function print_debitnote(DebitNote $debitnote)
  {
    $html = view('focus.debitnotes.print_debitnote', ['resource' => $debitnote])->render();
    $pdf = new \Mpdf\Mpdf(config('pdf'));
    $pdf->WriteHTML($html);
    $headers = array(
      "Content-type" => "application/pdf",
      "Pragma" => "no-cache",
      "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
      "Expires" => "0"
    );

    return Response::stream($pdf->Output('debitnote.pdf', 'I'), 200, $headers);
  }

  /**
   * Customer Invoices
   */
  public function customer_invoice()
  {
    $date = '';
    $last_month_day = (new DateTime($date))->format('Y-m-t');
    $date_6months_prior = date('Y-m-d', strtotime("{$last_month_day} -6 months"));
    $first_month_day = substr_replace($date_6months_prior, '01', -2, 2);
  }

  /**
   * Return Goods
   */
  public function return_goods(Request $request)
  {
    $products = ProductVariation::whereHas('product', fn($q) =>  $q->whereHas('unit'))
      ->limit(10)->get()
      ->map(function ($v) {
        $mod_v = $v->replicate();
        $mod_v['id'] = $v->id;
        if ($v->product && $v->product->unit) {
          $mod_v['unit'] = $v->product->unit->code;
        }

        return $mod_v;
      });

    return response()->json($products);
  }
}
