<?php
/*
 * Rose Business Suite - Accounting, CRM and POS Software
 * Copyright (c) UltimateKode.com. All Rights Reserved
 * ***********************************************************************
 *
 *  Email: support@ultimatekode.com
 *  Website: https://www.ultimatekode.com
 *
 *  ************************************************************************
 *  * This software is furnished under a license and may be used and copied
 *  * only  in  accordance  with  the  terms  of such  license and with the
 *  * inclusion of the above copyright notice.
 *  * If you Purchased from Codecanyon, Please read the full License from
 *  * here- http://codecanyon.net/licenses/standard/
 * ***********************************************************************
 */

namespace App\Http\Controllers\Focus\customer;

use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;
use App\Repositories\Focus\customer\CustomerRepository;
use App\Http\Requests\Focus\customer\ManageCustomerRequest;

/**
 * Class CustomersTableController.
 */
class CustomersTableController extends Controller
{
    /**
     * variable to store the repository object
     * @var CustomerRepository
     */
    protected $customer;
    protected $balance = 0;

    /**
     * contructor to initialize repository object
     * @param CustomerRepository $customer ;
     */
    public function __construct(CustomerRepository $customer)
    {
        $this->customer = $customer;
    }

    /**
     * This method return the data of the model
     * @param ManageCustomerRequest $request
     *
     * @return mixed
     */
    public function __invoke(ManageCustomerRequest $request)
    {
        if (request('is_transaction')) return $this->invoke_transaction();
        if (request('is_invoice')) return $this->invoke_invoice();
        if (request('is_statement')) return $this->invoke_statement();
            
        $core = $this->customer->getForDataTable();
        return Datatables::of($core)
            ->escapeColumns(['id'])
            ->addIndexColumn()
            ->addColumn('company', function ($customer) {
                $customer_name = $customer->company? $customer->company :  $customer->name;
                return '<a class="font-weight-bold" href="' . route('biller.customers.show', $customer) . '">' . $customer_name . '</a>';
            })
            ->make(true);
    }

    /**
     * Statement on Account
     */
    public function invoke_transaction()
    {
        $core = $this->customer->getTransactionsForDataTable();
        
        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($tr) {
            return '<span sortid="'. strtotime($tr->date) .'">'. dateFormat($tr->date) .'</span>';
        })
        ->addColumn('type', function ($tr) {
            return $tr->origin_type;
        })
        ->addColumn('note', function ($tr) {
            $note = '';
            switch ($tr->origin_type) {
                case 'manual_journal': $note = "({$tr->manual_journal->tid_code}) {$tr->manual_journal->note}"; break;
                case 'invoice': $note = "({$tr->invoice->tid_code}) {$tr->invoice->note}"; break;
                case 'deposit': $note = "({$tr->deposit->tid_code}) {$tr->deposit->note}"; break;
                case 'credit_note': $note = "({$tr->creditnote->tid_code}) {$tr->creditnote->note}"; break;
            }
            return $note;
        })
        ->addColumn('invoice_amount', function ($tr) {
            $debit = $tr->frx_rate > 1? $tr->frx_debit : $tr->debit;
            return numberFormat($debit);
        })
        ->addColumn('amount_paid', function ($tr) {
            $credit = $tr->frx_rate > 1? $tr->frx_credit : $tr->credit;
            return numberFormat($credit);
        })
        ->addColumn('account_balance', function ($tr) {
            $debit = $tr->frx_rate > 1? $tr->frx_debit : $tr->debit;
            $credit = $tr->frx_rate > 1? $tr->frx_credit : $tr->credit;
            if ($debit > 0) $this->balance += $debit;
            elseif ($credit > 0) $this->balance -= $credit;

            return numberFormat($this->balance);
        })
        ->make(true);
    }

    /**
     * Invoices report
     *  */ 
    public function invoke_invoice()
    {
        $core = $this->customer->getInvoicesForDataTable();

        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($invoice) {
            return dateFormat($invoice->date);
        })
        ->addColumn('status', function ($invoice) {
            return $invoice->status;
        })
        ->addColumn('note', function ($invoice) {
            return "({$invoice->tid_code}) {$invoice->note}";
        })
        ->addColumn('amount', function ($invoice) {
            return numberFormat($invoice->total);
        })
        ->addColumn('paid', function ($invoice) {
            return numberFormat($invoice->amount_received);
        })
        ->make(true);
    }

    /**
     * Statement on Invoice
     */
    public function invoke_statement()
    {
        $core = $this->customer->getStatementForDataTable();
        
        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($statement) {
            return dateFormat($statement->date);
        })
        ->addColumn('type', function ($statement) {
            $type = $statement->type;
            if ($type == 'invoice') $type = '<a href="'. route('biller.invoices.show', $statement->invoice_id) .'">'. $type .'</a>';
            if ($type == 'credit_note') $type = '<a href="'. route('biller.creditnotes.show', $statement->creditnote_id) .'">'. $type .'</a>';

            return $type;
        })
        ->addColumn('note', function ($statement) {
            return $statement->note;
        })
        ->addColumn('invoice_amount', function ($statement) {
            return numberFormat($statement->debit);
        })
        ->addColumn('amount_paid', function ($statement) {
            return numberFormat($statement->credit);
        })
        ->addColumn('invoice_balance', function ($statement) {
            if ($statement->type == 'invoice') 
                $this->balance = $statement->debit;
            else $this->balance -= $statement->credit;
            return numberFormat($this->balance);
        })
        ->make(true);
    }
}
